#ifndef _MATH3D_H_
#define _MATH3D_H_

// rozne geometricke a matematicke funkcie *********************************

// systemove kniznice ******************************************************
#include <cmath>

// minimum a maximum hodnot
#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
template <class T>
static inline T min(T a,T b) { return(a < b ? a : b); }
template <class T>
static inline T max(T a,T b) { return(a > b ? a : b); }

// konstanta PI
#ifndef PI
#define PI          3.14159265359
#endif
#define PI_MUL_2    6.28318530718
#define PI_DIV_2    1.57079632679
#define PI_DIV_180  0.01745329251
#define _180_DIV_PI 57.2957795131

// prevod stupne<->radiany
#define DEG2RAD(a) (PI_DIV_180*(a))
#define RAD2DEG(a) (_180_DIV_PI*(a))

// 2-rozmerny vektor *******************************************************
class vec2 {
public:
 // udaje ------------------------------------------------------------------
 union {
  struct {
   float x,y; // prvky vektora
  };
  float d[2]; // prvky vektora
 };

 // konstruktory -----------------------------------------------------------
 vec2():x(0.0f),y(0.0f) {}
 vec2(float A,float B):x(A),y(B) {}

 // operatory --------------------------------------------------------------
 // index
 float &operator[](const int I) { return(d[I]); }
 // odcitanie vektorov
 vec2 operator-(const vec2 V) { return(vec2(x-V.x,y-V.y)); }

 // funkcie ----------------------------------------------------------------
 // priradenie prvkov vektoru
 void set(float A,float B) { x=A; y=B; }
};

// 3-rozmerny vektor *******************************************************
class vec3 {
public:
 // udaje ------------------------------------------------------------------
 union {
  struct {
   float x,y,z; // prvky vektora
  };
  float d[3];   // prvky vektora
 };

 // konstruktory -----------------------------------------------------------
 vec3():x(0.0f),y(0.0f),z(0.0f) {}
 vec3(float A,float B,float C):x(A),y(B),z(C) {}

 // operatory --------------------------------------------------------------
 // index
 float &operator[](const int I) { return(d[I]); }
 // pricitanie k vektoru
 vec3 operator+=(const vec3 V) { x+=V.x; y+=V.y; z+=V.z; return(*this); }
 // scitanie vektorov
 vec3 operator+(const vec3 V) { return(vec3(x+V.x,y+V.y,z+V.z)); }
 // odcitanie vektorov
 vec3 operator-(const vec3 V) { return(vec3(x-V.x,y-V.y,z-V.z)); }
 // nasobenie skalarom
 vec3 operator*(const float S) { return(vec3(x*S,y*S,z*S)); }
 // delenie skalarom
 vec3 operator/(const float S) {
  recip=1.0f; if (S!=0.0f) recip=1.0f/S;
  return(vec3(x*recip,y*recip,z*recip));
 }

 // funkcie ----------------------------------------------------------------
 // priradenie prvkov vektoru
 void set(float A,float B,float C) { x=A; y=B; z=C; }
 // delenie skalarom
 vec3 div(const float S) {
  float recip=1.0f; if (S!=0.0f) recip=1.0f/S;
  x*=recip; y*=recip; z*=recip; return(*this);
 }
 // skalarny sucin
 float dot(const vec3 V) { return(x*V.x+y*V.y+z*V.z); }
 // vektorovy sucin
 vec3 cross(const vec3 V) { return(vec3(y*V.z-z*V.y,z*V.x-x*V.z,x*V.y-y*V.x)); }
 // (dlzka vektora)^2
 float squaredlen(void) { return(x*x+y*y+z*z); }
 // dlzka vektora
 float magnitude(void) { return(sqrtf(squaredlen())); }
 // normalizacia vektora
 vec3 normalize(void) { div(magnitude()); return(*this); }
 // projekcia
 vec3 proj(vec3 A,vec3 B) {
  float p=A.dot(B);
  return(A*(p/A.squaredlen()));
 }

 const float *get_data(void) { return(d); }
private:
 float recip; // pomocna premenna
};

// 4-rozmerny vektor *******************************************************
class vec4 {
public:
 // udaje ------------------------------------------------------------------
 union {
  struct {
   float x,y,z,w; // prvky vektora
  };
  float d[4];     // prvky vektora
 };

 // konstruktory -----------------------------------------------------------
 vec4():x(0.0f),y(0.0f),z(0.0f),w(0.0f) {}
 vec4(float A,float B,float C,float D):x(A),y(B),z(C),w(D) {}

 // operatory --------------------------------------------------------------
 // index
 float &operator[](const int I) { return(d[I]); }

 // funkcie ----------------------------------------------------------------
 // priradenie prvkov vektoru
 void set(float A,float B,float C,float D) { x=A; y=B; z=C; w=D; }

 const float *get_data(void) { return(d); }
};

// matica typu 4x4 *********************************************************
class mat4x4 {
public:
 // udaje ------------------------------------------------------------------
 float d[16];

 // konstruktory -----------------------------------------------------------
 mat4x4() {
  d[0]=1.0f; d[4]=0.0f; d[8] =0.0f; d[12]=0.0f;
  d[1]=0.0f; d[5]=1.0f; d[9] =0.0f; d[13]=0.0f;
  d[2]=0.0f; d[6]=0.0f; d[10]=1.0f; d[14]=0.0f;
  d[3]=0.0f; d[7]=0.0f; d[11]=0.0f; d[15]=1.0f;
 }
 mat4x4(const mat4x4 &M) { (*this)=M; } // ak M zanikne, zanikne aj instancia mat4x4
 mat4x4(const float *M) { memcpy(d,M,sizeof(float)*16); } // ak M zanikne, instancia mat4x4 zostava

 // operatory --------------------------------------------------------------
 // index
 float &operator[](const int I) { return(d[I]); }
 // nasobenie matic, vysledok sa ulozi do novej matice M2 = M1 * M
 mat4x4 operator*(const mat4x4 M) {
  mat4x4 m; // pomocna matica
  m[0] =d[0]*M.d[0] +d[4]*M.d[1] +d[8] *M.d[2] +d[12]*M.d[3];
  m[1] =d[1]*M.d[0] +d[5]*M.d[1] +d[9] *M.d[2] +d[13]*M.d[3];
  m[2] =d[2]*M.d[0] +d[6]*M.d[1] +d[10]*M.d[2] +d[14]*M.d[3];
  m[3] =d[3]*M.d[0] +d[7]*M.d[1] +d[11]*M.d[2] +d[15]*M.d[3];
  m[4] =d[0]*M.d[4] +d[4]*M.d[5] +d[8] *M.d[6] +d[12]*M.d[7];
  m[5] =d[1]*M.d[4] +d[5]*M.d[5] +d[9] *M.d[6] +d[13]*M.d[7];
  m[6] =d[2]*M.d[4] +d[6]*M.d[5] +d[10]*M.d[6] +d[14]*M.d[7];
  m[7] =d[3]*M.d[4] +d[7]*M.d[5] +d[11]*M.d[6] +d[15]*M.d[7];
  m[8] =d[0]*M.d[8] +d[4]*M.d[9] +d[8] *M.d[10]+d[12]*M.d[11];
  m[9] =d[1]*M.d[8] +d[5]*M.d[9] +d[9] *M.d[10]+d[13]*M.d[11];
  m[10]=d[2]*M.d[8] +d[6]*M.d[9] +d[10]*M.d[10]+d[14]*M.d[11];
  m[11]=d[3]*M.d[8] +d[7]*M.d[9] +d[11]*M.d[10]+d[15]*M.d[11];
  m[12]=d[0]*M.d[12]+d[4]*M.d[13]+d[8] *M.d[14]+d[12]*M.d[15];
  m[13]=d[1]*M.d[12]+d[5]*M.d[13]+d[9] *M.d[14]+d[13]*M.d[15];
  m[14]=d[2]*M.d[12]+d[6]*M.d[13]+d[10]*M.d[14]+d[14]*M.d[15];
  m[15]=d[3]*M.d[12]+d[7]*M.d[13]+d[11]*M.d[14]+d[15]*M.d[15];
  return(m);
 }
 // nasobenie matic, vysledok sa ulozi do povodnej matice M1 = M1 * M
 mat4x4 operator*=(const mat4x4 M) {
  mat4x4 m; // pomocna matica
  m[0] =d[0]*M.d[0] +d[4]*M.d[1] +d[8] *M.d[2] +d[12]*M.d[3];
  m[1] =d[1]*M.d[0] +d[5]*M.d[1] +d[9] *M.d[2] +d[13]*M.d[3];
  m[2] =d[2]*M.d[0] +d[6]*M.d[1] +d[10]*M.d[2] +d[14]*M.d[3];
  m[3] =d[3]*M.d[0] +d[7]*M.d[1] +d[11]*M.d[2] +d[15]*M.d[3];
  m[4] =d[0]*M.d[4] +d[4]*M.d[5] +d[8] *M.d[6] +d[12]*M.d[7];
  m[5] =d[1]*M.d[4] +d[5]*M.d[5] +d[9] *M.d[6] +d[13]*M.d[7];
  m[6] =d[2]*M.d[4] +d[6]*M.d[5] +d[10]*M.d[6] +d[14]*M.d[7];
  m[7] =d[3]*M.d[4] +d[7]*M.d[5] +d[11]*M.d[6] +d[15]*M.d[7];
  m[8] =d[0]*M.d[8] +d[4]*M.d[9] +d[8] *M.d[10]+d[12]*M.d[11];
  m[9] =d[1]*M.d[8] +d[5]*M.d[9] +d[9] *M.d[10]+d[13]*M.d[11];
  m[10]=d[2]*M.d[8] +d[6]*M.d[9] +d[10]*M.d[10]+d[14]*M.d[11];
  m[11]=d[3]*M.d[8] +d[7]*M.d[9] +d[11]*M.d[10]+d[15]*M.d[11];
  m[12]=d[0]*M.d[12]+d[4]*M.d[13]+d[8] *M.d[14]+d[12]*M.d[15];
  m[13]=d[1]*M.d[12]+d[5]*M.d[13]+d[9] *M.d[14]+d[13]*M.d[15];
  m[14]=d[2]*M.d[12]+d[6]*M.d[13]+d[10]*M.d[14]+d[14]*M.d[15];
  m[15]=d[3]*M.d[12]+d[7]*M.d[13]+d[11]*M.d[14]+d[15]*M.d[15];
  (*this)=m;
  return(*this);
 }
 // nasobenie matice skalarom, vysledok sa ulozi do novej matice M2 = M1 * S
 mat4x4 operator*(const float S) {
  mat4x4 m; // pomocna matica
  loopi(16) m[i]=m[i]*S;
  return(m);
 }
 // nasobenie matice a vektora v2 = M1 * v1
 vec3 operator*(const vec3 V) {
  a=d[0]*V.x+d[4]*V.y+d[8]*V.z+ d[12];
  b=d[1]*V.x+d[5]*V.y+d[9]*V.z+ d[13];
  c=d[2]*V.x+d[6]*V.y+d[10]*V.z+d[14];
  return(vec3(a,b,c));
 }
 // scitanie matic, vysledok sa ulozi do novej matice M2 = M1 + M
 mat4x4 operator+(mat4x4 M) {
  mat4x4 m; // pomocna matica
  loopi(16) m[i]=m[i]+M[i];
  return(m);
 }
 // odcitanie matic, vysledok sa ulozi do novej matice M2 = M1 + M
 mat4x4 operator-(mat4x4 M) {
  mat4x4 m=M*(-1.0f);
  return((*this)+m);
 }

 // funkcie ----------------------------------------------------------------
 // priradenie prvkov matici
 void set(const float M[16]) {
  d[0]=M[0]; d[4]=M[4]; d[8] =M[8];  d[12]=M[12];
  d[1]=M[1]; d[5]=M[5]; d[9] =M[9];  d[13]=M[13];
  d[2]=M[2]; d[6]=M[6]; d[10]=M[10]; d[14]=M[14];
  d[3]=M[3]; d[7]=M[7]; d[11]=M[11]; d[15]=M[15];
 }
 void load(float *A) { memcpy(this->d,A,sizeof(float)*16); }
 // ziskanie prvkov matice
 const float *get_data(void) { return(d); }
 // jednotkova matica
 void identity(void) {
  d[0]=1.0f; d[4]=0.0f; d[8] =0.0f; d[12]=0.0f;
  d[1]=0.0f; d[5]=1.0f; d[9] =0.0f; d[13]=0.0f;
  d[2]=0.0f; d[6]=0.0f; d[10]=1.0f; d[14]=0.0f;
  d[3]=0.0f; d[7]=0.0f; d[11]=0.0f; d[15]=1.0f;
 }
 // transponovana matica
 void transpose(void) {
  tmp=d[4];  d[4] =d[1];  d[1] =tmp;
  tmp=d[8];  d[8] =d[2];  d[2] =tmp;
  tmp=d[12]; d[12]=d[3];  d[3] =tmp;
  tmp=d[9];  d[9] =d[6];  d[6] =tmp;
  tmp=d[13]; d[13]=d[7];  d[7] =tmp;
  tmp=d[14]; d[14]=d[11]; d[11]=tmp;
 }
 // posun
 void translate(float X,float Y,float Z) {
  mat4x4 m; // pomocna matica
  m[12]=X; m[13]=Y; m[14]=Z;
  (*this)*=m;
 }
 // rotacia o definovane uhly
 void rotate(float A,float B,float C) {
  cx=cosf(A); sx=sinf(A);
  cy=cosf(B); sy=sinf(B);
  cz=cosf(C); sz=sinf(C);

  mat4x4 m; // pomocna matica
  m[0]=cy*cz; m[4]=(sx*sy*cz)-(cx*sz); m[8] =(cx*sy*cz)+(sx*sz);
  m[1]=cy*sz; m[5]=(sx*sy*sz)+(cx*cz); m[9] =(cx*sy*sz)-(sx*cz);
  m[2]=-sy;   m[6]=sx*cy;              m[10]=cx*cy;
  (*this)*=m;
 }
 // rotacia o uhol A okolo osi X Y Z
 void rotate(float A,float X,float Y,float Z) {
  c=cos(A); c1=1.0f-c; s=sin(A);
  l=sqrtf(X*X+Y*Y+Z*Z);
  X/=l; Y/=l; Z/=l;

  mat4x4 m; // pomocna matica
  m[0]=X*X*c1+c;   m[4]=Y*X*c1-Z*s; m[8] =Z*X*c1+Y*s;
  m[1]=X*Y*c1+Z*s; m[5]=Y*Y*c1+c;   m[9] =Z*Y*c1-X*s;
  m[2]=X*Z*c1-Y*s; m[6]=Y*Z*c1+X*s; m[10]=Z*Z*c1+c;
  (*this)*=m;
 }
 // skalovanie
 void scale(float X,float Y,float Z) {
  mat4x4 m; // pomocna matica
  m[0]=X; m[5]=Y; m[10]=Z;
  (*this)*=m;
 }
 // lookat matica
 void lookat(float eyeX,float eyeY,float eyeZ,
             float centerX,float centerY,float centerZ,
             float upX=0.0f,float upY=1.0f,float upZ=0.0f) {
  float F[3]={centerX-eyeX,centerY-eyeY,centerZ-eyeZ};
  float Fmag=sqrtf(F[0]*F[0]+F[1]*F[1]+F[2]*F[2]);
  float f[3]={F[0]/Fmag,F[1]/Fmag,F[2]/Fmag};
  float upmag=sqrtf(upX*upX+upY*upY+upZ*upZ);
  float up[3]={upX/upmag,upY/upmag,upZ/upmag};

  mat4x4 m; // pomocna matica
  m[0]=f[1]*up[2]-f[2]*up[1]; m[4]=f[2]*up[0]-f[0]*up[2]; m[8] =f[0]*up[1]-f[1]*up[0];
  m[1]=m[4]*f[2]-m[8]*f[1];   m[5]=m[8]*f[0]-m[0]*f[2];   m[9] =m[0]*f[1]-m[4]*f[0];
  m[2]=-f[0];                 m[6]=-f[1];                 m[10]=-f[2];
  (*this)*=m;

  translate(-eyeX,-eyeY,-eyeZ);
 }
 // frustum matica
 void frustrum(float left,float right,float bottom,float top,float near,float far) {
  mat4x4 m; // pomocna matica
  m[0] =2.0f*near/(right-left);
  m[5] =2.0f*near/(top-bottom);
  m[8] =(right+left)/(right-left);
  m[9] =(top+bottom)/(top-bottom);
  m[10]=-((far+near)/(far-near));
  m[11]=-1.0f;
  m[14]=-(2.0f*far*near/(far-near));
  m[15]=0.0f;
  (*this)*=m;
 }
 // ortho matica
 void ortho(float left,float right,float bottom,float top,float near,float far) {
  mat4x4 m; // pomocna matica
  m[0] =2.0f/(right-left);
  m[5] =2.0f/(top-bottom);
  m[10]=-2.0f/(far-near);
  m[12]=-(right+left)/(right-left);
  m[13]=-(top+bottom)/(top-bottom);
  m[14]=-(far+near)/(far-near);
  (*this)*=m;
 }
 // perspective matica
 void perspective(float fov,float aspect,float near,float far) {
  fov=DEG2RAD(fov);
  float f=cos(fov/2.0f)/sin(fov/2.0f);
  mat4x4 m; // pomocna matica
  m[0] =f/aspect;
  m[5] =f;
  m[10]=(far+near)/(near-far);
  m[11]=-1.0f;
  m[14]=2.0f*far*near/(near-far);
  m[15]=0.0f;
  (*this)*=m;
 }
 // determinant matice 4x4
 float det(void) {
  // Laplaceova veta
  // minor = determinant matice, ktora vznikla z povodnej matice vynechanim
  // i-teho riadku a j-teho stlpca
  // algebraicky doplnok = kofaktor = (-1)^(i+j) * minor_ij
  // detM = sucet j=1 az n [ kofaktor * prvok_matice_ij ]
  //      = sucet j=1 az n [ (-1)^(i+j) * minor_ij * prvok_matice_ij ]
  //                  5  9 13                  1  9 13                  1 5 13                  1 5  9
  // detM4x4 = -1^2 * 6 10 14 * pm_11 + -1^3 * 2 10 14 * pm_12 + -1^4 * 2 6 14 * pm_13 + -1^5 * 2 6 10 * pm_14
  //                  7 11 15                  3 11 15                  3 7 15                  3 7 11
  // Sarusovo (krizove) pravidlo pre vypocet determinantu matice 3x3
  // 0  3  6
  // 1  4  7
  // 2  5  8
  // detM = 0 4 8 + 1 5 6 + 2 3 7 - 6 4 2 - 7 5 0 - 8 3 1

  // minory z matic 3x3
  float a1=d[10]*d[15]-d[11]*d[14];
  float a2=d[9]*d[15]-d[11]*d[13];
  float a3=d[9]*d[14]-d[10]*d[13];
  float a4=d[7]*d[2]-d[6]*d[3];
  float a5=d[7]*d[1]-d[5]*d[3];
  float a6=d[6]*d[1]-d[5]*d[2];
  // detM4x4
  return(d[0] *(d[5] *a1-d[6] *a2+d[7] *a3)-
         d[4] *(d[1] *a1-d[2] *a2+d[3] *a3)+
         d[8] *(d[13]*a4-d[14]*a5+d[15]*a6)-
         d[12]*(d[9] *a4-d[10]*a5+d[11]*a6));
 }
 // inverzna matica - vysledok sa zapise do novej matice
 mat4x4 inverse(void) {
  // A_adj - adjungovana matica zlozena z algebraickych doplnkov
  // algebraicke doplnky su v adjungovanej matici ulozene transponovane
  // prvok ij adjungovanej matice = Aji
  //
  // M_inv = A_adj / detM - iverzna matica
  //
  // matica algebraickych doplnkov - kofaktorov = determinanty matic 3x3
  float minor[16];
  mat4x4 m; // pomocna matica
  minor[0] =d[5]*d[10]*d[15]+d[6]*d[11]*d[13]+d[7]*d[9]*d[14]-    // 0 riadok 0 stlpec
            d[7]*d[10]*d[13]-d[6]*d[9]*d[15]-d[5]*d[11]*d[14];
  minor[1] =-(d[1]*d[10]*d[15]+d[2]*d[11]*d[13]+d[3]*d[9]*d[14]-  // 0 riadok 4 stlpec
              d[3]*d[10]*d[13]-d[2]*d[9]*d[15]-d[1]*d[11]*d[14]);
  minor[2] =d[1]*d[6]*d[15]+d[2]*d[7]*d[13]+d[3]*d[5]*d[14]-      // 0 riadok 8 stlpec
            d[3]*d[6]*d[13]-d[2]*d[5]*d[15]-d[1]*d[7]*d[14];
  minor[3] =-(d[1]*d[6]*d[11]+d[2]*d[7]*d[9]+d[3]*d[5]*d[10]-     // 0 riadok 12 stlpec
              d[3]*d[6]*d[9]-d[2]*d[5]*d[11]-d[1]*d[7]*d[10]);
  minor[4] =-(d[4]*d[10]*d[15]+d[6]*d[11]*d[12]+d[7]*d[8]*d[14]-  // 1 riadok 0 stlpec
              d[7]*d[10]*d[12]-d[6]*d[8]*d[15]-d[4]*d[11]*d[14]);
  minor[5] =d[0]*d[10]*d[15]+d[2]*d[11]*d[12]+d[3]*d[8]*d[14]-    // 1 riadok 4 stlpec
            d[3]*d[10]*d[12]-d[2]*d[8]*d[15]-d[0]*d[11]*d[14];
  minor[6] =-(d[0]*d[6]*d[15]+d[2]*d[7]*d[12]+d[3]*d[4]*d[14]-    // 1 riadok 8 stlpec
              d[3]*d[6]*d[12]-d[2]*d[4]*d[15]-d[0]*d[7]*d[14]);
  minor[7] =d[0]*d[6]*d[11]+d[2]*d[7]*d[8]+d[3]*d[4]*d[10]-       // 1 riadok 12 stlpec
            d[3]*d[6]*d[8]-d[2]*d[4]*d[11]-d[0]*d[7]*d[10];
  minor[8] =d[4]*d[9]*d[15]+d[5]*d[11]*d[12]+d[7]*d[8]*d[13]-     // 2 riadok 0 stlpec
            d[7]*d[9]*d[12]-d[5]*d[8]*d[15]-d[4]*d[11]*d[13];
  minor[9] =-(d[0]*d[9]*d[15]+d[1]*d[11]*d[12]+d[3]*d[8]*d[13]-   // 2 riadok 4 stlpec
              d[3]*d[9]*d[12]-d[1]*d[8]*d[15]-d[0]*d[11]*d[13]);
  minor[10]=d[0]*d[5]*d[15]+d[1]*d[7]*d[12]+d[3]*d[4]*d[13]-      // 2 riadok 8 stlpec
            d[3]*d[5]*d[12]-d[1]*d[4]*d[15]-d[0]*d[7]*d[13];
  minor[11]=-(d[0]*d[5]*d[11]+d[1]*d[7]*d[8]+d[3]*d[4]*d[9]-      // 2 riadok 12 stlpec
              d[3]*d[5]*d[8]-d[1]*d[4]*d[11]-d[0]*d[7]*d[9]);
  minor[12]=-(d[4]*d[9]*d[14]+d[5]*d[10]*d[12]+d[6]*d[8]*d[13]-   // 3 riadok 0 stlpec
              d[6]*d[9]*d[12]-d[5]*d[8]*d[14]-d[4]*d[10]*d[13]);
  minor[13]=d[0]*d[9]*d[14]+d[1]*d[10]*d[12]+d[2]*d[8]*d[13]-     // 3 riadok 4 stlpec
            d[2]*d[9]*d[12]-d[1]*d[8]*d[14]-d[0]*d[10]*d[13];
  minor[14]=-(d[0]*d[5]*d[14]+d[1]*d[6]*d[12]+d[2]*d[4]*d[13]-    // 3 riadok 8 stlpec
              d[2]*d[5]*d[12]-d[1]*d[4]*d[14]-d[0]*d[6]*d[13]);
  minor[15]=d[0]*d[5]*d[10]+d[1]*d[6]*d[8]+d[2]*d[4]*d[9]-        // 3 riadok 12 stlpec
            d[2]*d[5]*d[8]-d[1]*d[4]*d[10]-d[0]*d[6]*d[9];

  float d=det();
  if (d!=0.0f) { loopi(16) m[i]=minor[i]/d; }
  return(m);
 }
 // ortogonalna matica - 3 stlpce matice = 3 vektory su navzajom kolme a normalizovane
 // A At = I
 // A^(-1) = At
 // detA = 1 alebo -1
 // Gram–Schmidt ortogonalizacia
 // uk = vk - sucet j=1 az k-1 [ proj_uj(vk) ]  uk = uk / | uk |
 // proj_uj(v) = u * ( v . u ) / ( u . u )
 void orthogonalize(void) {
  vec3 v1(d[0],d[1],d[2]);
  vec3 v2(d[4],d[5],d[6]);
  vec3 v3(d[8],d[9],d[10]);

  vec3 p;
  vec3 u1=v1.normalize();
  vec3 u2=(v2-p.proj(u1,v2)); u2.normalize();
  vec3 u3=(v3-p.proj(u1,v3)-p.proj(u2,v3)); u3.normalize();

  d[0]=u1[0]; d[4]=u2[0]; d[8] =u3[0]; d[12]=0.0f;
  d[1]=u1[1]; d[5]=u2[1]; d[9] =u3[1]; d[13]=0.0f;
  d[2]=u1[2]; d[6]=u2[2]; d[10]=u3[2]; d[14]=0.0f;
  d[3]=0.0f;  d[7]=0.0f;  d[11]=0.0f;  d[15]=1.0f;
 }
private:
 float tmp; // pomocna premenna
 float cx,cy,cz,sx,sy,sz; // pomocne premenne
 float a,b,c,c1,s,l;
};

// trieda premennych a funkcii pre pracu s matematikou *********************
class Cmath3d {
public:
 void calc_tbn(vec3 V0,vec3 V1,vec3 V2,vec2 V0T,vec2 V1T,vec2 V2T,
               vec3 &T,vec3 &B,vec3 &N);
private:
 vec3 v1v0,v2v0;
 vec2 t1t0,t2t0;
 float recip;
};

#endif
