from scapy.all import *  # Scapy
from scapy.layers.dns import DNS, DNSQR
from scapy.layers.inet import IP, TCP, ICMP, UDP
from scapy.layers.l2 import Ether, ARP

from dash import html, callback
import dash_bootstrap_components as dbc
from dash import dash_table
from dash import dcc
from dash.dependencies import Input, Output, State

import pandas as pd  # Pandas

PORTSVC = {
    7: "ECHO protokol",
    13: "DAYTIME protokol",
    20: "FTP prenos dát",
    21: "FTP správa",
    22: "SSH protokol",
    23: "Telnet protokol",
    25: "SMTP protokol",
    37: "TIME protokol",
    42: "WINS služba",
    43: "WHOIS protokol",
    52: "XNS Time protokol",
    53: "DNS služba",
    56: "RAP protokol",
    67: "DHCP protokol",
    68: "DHCP protokol",
    69: "TFTP protokol",
    79: "FINGER protokol",
    80: "HTTP protokol",
    88: "Kerberos autentifikácia",
    107: "Vzdialená Telnet služba",
    109: "POP2 protokol",
    110: "POP3 protokol",
    115: "SFTP protokol",
    118: "SQL služby",
    123: "NTP protokol",
    137: "NetBIOS Name služba",
    138: "NetBIOS Datagram služba",
    139: "NetBIOS Session služba",
    143: "IMAP protokol",
    152: "BFTP Background File Transfer Program",
    156: "SQL služba",
    161: "SNMP protokol",
    162: "SNMP protokol",
    170: "Network PostScript",
    194: "IRC služba",
    220: "IMAPv3",
    280: "HTTP manažment",
    389: "LDAP protokol",
    443: "HTTPS protokol",
    445: "Server Message Block (SMB)",
    464: "Kerberos nastavenie/zmena hesla",
    465: "SMTP cez TLS/SSL",
    500: "ISAKMP/IKE",
    520: "RIP protokol",
    530: "RPC protokol",
    543: "klogin Kerberos login",
    544: "kshell Kerberos shell",
    546: "DHCPv6 klient",
    547: "DHCPv6 server",
    554: "RTSP protokol",
    631: "CUPS služba",
    636: "LDAP cez TLS/SSL",
    901: "Samba WAT",
    903: "VMware konzola",
    989: "FTP data cez TLS/SSL",
    990: "FTP kontrola cez TLS/SSL",
    995: "POP3 cez TLS/SSL",
    3306: "MySQL databáza",
    5432: "PostgreSQL databáza",
    8080: "HTTP protokol",
    8443: "HTTPS protokol"
}

# porty, ktore sa budu skenovat
ports_to_scan = [20, 21, 22, 23, 25, 53, 67, 68, 80, 110, 143, 443, 8080, 8443]

# inicializacia Pandas DataFrame, index=id
answers_data = pd.DataFrame(columns=['id', 'rowid', 'scanTYPE', 'dstMAC', 'dstIP'])
answers_data.set_index('id', inplace=True)
selected_data = pd.DataFrame(columns=['id', 'rowid', 'dstPORT', 'dstSVC'])
selected_data.set_index('id', inplace=True)

pkt_data = pd.DataFrame(columns=['id', 'pktID', 'srcIP_PORT', 'srcMAC', 'dstIP_PORT', 'dstMAC'])
pkt_data.set_index("id", inplace=True)


# SKENUJ SIET
def MAC_IP_scan(host):
    # id prijatych odoziev
    m_id = 1
    # zmazanie DataFramov
    answers_data.drop(answers_data.index, inplace=True)
    selected_data.drop(selected_data.index, inplace=True)

    # ARP scan
    # sendrcv() vrati 2x List - ans = SndRcvList, unans = PacketList
    # srp - send Ethernet frame na 2. vrstve OSI modelu - Ethernet header / ARP header
    ans, unans = srp(Ether(dst="ff:ff:ff:ff:ff:ff") / ARP(pdst=host), timeout=3)
    # print('ans: ', ans.summary())
    # print('unans:  ', unans.summary())
    for s, r in ans:
        # ans = SndRcvList = send, received frames
        # print('s: ', s)
        # print('r:  ', r)
        answers_data.loc[m_id] = (m_id, 'ARP', r.hwsrc, r.psrc)
        m_id += 1

    # ICMP Ping scan
    # ans, unans = sr(IP(dst=host) / ICMP(), timeout=3)
    # for s, r in ans:
    #    answers_data.loc[m_id] = (m_id, 'ICMP Ping', '-', r.sprintf('%IP.src%'))
    #    m_id += 1

    # TCP Ping
    # ans, unans = sr(IP(dst=host) / TCP(dport=80, flags="S"))
    # for s, r in ans:
    #    answers_data.loc[m_id] = (m_id, 'TCP Ping', '-', r.sprintf('%IP.src%'))
    #    m_id += 1

    # UDP Ping
    # ans, unans = sr(IP(dst=host) / UDP(dport=0))
    # for s, r in ans:
    #    answers_data.loc[m_id] = (m_id, 'UDP Ping', '-', r.sprintf('%IP.src%'))
    #    m_id += 1


# SKENUJ PORTY VYBRANEJ IP ADRESY
def PORT_scan(host):
    # id prijatych odoziev
    m_id = 1
    # zmazanie DataFramu
    selected_data.drop(selected_data.index, inplace=True)

    # sken otvorenych portov
    # sr - send Network packet na 3. vrstve OSI modelu - IP header / TCP header
    ans, unans = sr(IP(dst=host) / TCP(sport=11111, dport=ports_to_scan, flags='S'), timeout=3, verbose=1)
    # print(ans.summary())
    for s, r in ans:
        # ak ma prijaty paket TCP vrstvu
        if r.haslayer(TCP):
            # ak srcPORT == dstPORT a TCP flags == SYN, ACK
            if s[TCP].dport == r[TCP].sport and r[TCP].flags == 'SA':
                try:
                    if r[TCP].sport in PORTSVC.keys():
                        svc = PORTSVC[r[TCP].sport]
                    else:
                        svc = str(r[TCP].sport)
                except AttributeError:
                    svc = 'Neznáma služba'
                selected_data.loc[m_id] = (m_id, r[TCP].sport, svc)
                m_id += 1
    # sken existencie DNS servera
    # sr - send Network packet na 3. vrstve OSI modelu - IP header / UDP header / DNS header
    ans, unans = sr(IP(dst=host) / UDP(dport=53) / DNS(rd=1, qd=DNSQR(qname='softengine.sk')), timeout=3, verbose=0)
    if ans and ans[UDP]:
        selected_data.loc[m_id] = (m_id, 53, 'DNS server')
        m_id += 1


myMAC = get_if_hwaddr(conf.iface)  # moja MAC adresa
myIP = get_if_addr(conf.iface)  # moja IP adresa


# Scapy Sniffer
class SnifferClass(Thread):
    def __init__(self, interface="eno1"):
        super().__init__()

        self.iface = interface  # vybrane rozhranie
        self.stop_sniffer = Event()  # pre spravne ukoncenie vlakna
        self.pkt_id = 0  # id prijateho paketu

    # spustenie funkcie pkt_recv v samostatnom vlakne
    def run(self):
        sniff(iface=self.iface, filter="ip", prn=self.pkt_recv, stop_filter=self.should_stop_sniffer)

    # ukoncenie vlakna
    def join(self, timeout=None):
        self.stop_sniffer.set()
        super().join(timeout)

    # spravne ukoncenie vlakna
    def should_stop_sniffer(self, pkt):
        return self.stop_sniffer.is_set()

    # Packets Sniffer
    def pkt_recv(self, pkt):
        if self.pkt_id > 500:  # ak bolo prijatych viac ako 500 paketov, DataFrame sa vymaze
            pkt_data.drop(pkt_data.index, inplace=True)
            self.pkt_id = 0

        # pkt.show()  # zobrazenie obsahu paketu

        # zdrojova a cielova IP adresa
        src_ip = ''
        dst_ip = ''
        # IP hlavicka
        if pkt.haslayer(IP):
            p = pkt[IP]
            src_ip = p.src
            dst_ip = p.dst

        # zdrojovy a cielovy PORT
        src_port = ''
        dst_port = ''
        # TCP hlavicka
        if pkt.haslayer(TCP):
            p = pkt[TCP]
            src_port = p.sport
            dst_port = p.dport

        # priprava textu IP:PORT pre zdroj
        if src_port != '':
            src_ip_port = str(src_ip) + ':' + str(src_port)
        else:
            src_ip_port = str(src_ip)
        # priprava textu IP:PORT pre ciel
        if dst_port != '':
            dst_ip_port = str(dst_ip) + ':' + str(dst_port)
        else:
            dst_ip_port = str(dst_ip)

        # ulozenie udajov do DataFrame
        pkt_data.loc[self.pkt_id] = (self.pkt_id, src_ip_port, pkt.src, dst_ip_port, pkt.dst)
        self.pkt_id += 1  # boli pridane udaje o pakete

        return

    def get_pkt_id(self):
        return self.pkt_id


my_sniffer = SnifferClass()  # objekt triedy SnifferClass
my_sniffer_running = 0  # vlakno je / nie je spustene
my_sniffer_progress = 0  # vizualne zobrazenie priebehu skenu

# vzhlad stranky
layout = dbc.Container([
    dbc.Row([
        html.H1('Sken 1 - Zdrojové a cieľové IP:PORT a MAC adresy')
    ]),
    dbc.Row([
        html.Table(
            style={'marginBottom': '20px'},
            children=[
                html.Tr(
                    children=[
                        html.Td(
                            style={'text-align': 'center', 'border-bottom': 'none',
                                   'border-right': '1px solid', 'width': '33%'},
                            children=[html.Div('Moja MAC adresa je:'),
                                      html.Div(className='ADResa', children=[myMAC])]),
                        html.Td(
                            style={'text-align': 'center', 'border-bottom': 'none',
                                   'border-right': '1px solid', 'width': '33%'},
                            children=[html.Div('Moja IP adresa je:'),
                                      html.Div(className='ADResa', children=[myIP])]),
                        html.Td(
                            style={'text-align': 'center', 'border-bottom': 'none', 'width': '33%'},
                            children=[html.Div('Skenuje sa sieť:'),
                                      html.Div(className='ADResa', children=[myIP + '/24'])])
                    ]
                )
            ]
        )
    ]),
    dbc.Row([
        html.Div(children=[html.Button(id='page1skenuj',
                                       children=['SKEN - START'],
                                       style={'margin-bottom': '10px',
                                              'background-color': '#333333',
                                              'color': '#ffffff'}
                                       ),
                           dcc.Input(id='page1pomocka', value='0', style={'display': 'none'}),
                           html.Div(children=[dbc.Progress(id='page1progbar', value=0, color='#1eaedb',
                                                           style={'height': '10px', 'margin-bottom': '10px'}),
                                              dcc.Interval(id='page1progbar-interval', n_intervals=0, interval=1000)])
                           ]),
        html.Div(children=[dash_table.DataTable(
            id='page1table1',
            fixed_rows={'headers': True, 'data': 0},
            columns=[{'id': 'pktID', 'name': 'ID'},
                     {'id': 'srcIP_PORT', 'name': 'Zdrojová IP:PORT'},
                     {'id': 'srcMAC', 'name': 'Zdrojová MAC'},
                     {'id': 'dstIP_PORT', 'name': 'Cieľová IP:PORT'},
                     {'id': 'dstMAC', 'name': 'Cieľová MAC'}],
            style_table={'border': 'grey solid', 'height': 600},
            style_header={'color': 'lightgrey',
                          'backgroundColor': 'rgb(30,30,30)',
                          'fontWeight': 'bold'},
            style_cell={'color': 'white',
                        'backgroundColor': 'rgb(50,50,50)',
                        'border': '0px solid',
                        'font-size': '1.2em',
                        'textAlign': 'center',
                        'min-width': '70px'},
            style_as_list_view=True,
            style_data_conditional=[{'if': {'row_index': 'odd'},
                                     'backgroundColor': 'rgb(70,70,70)'}],
            page_action='none'),
            dcc.Interval(
                id='page1table1_interval',
                interval=5000,  # aktualizuje n_intervals kazdych 5s
                n_intervals=0
            )
        ])
    ])
])


# Callback funkcia pre tlacidlo - potrebna pre spravne zobrazovanie textu tlacidla pri vybere stranok
@callback(Output('page1skenuj', 'children'),
          [Input('page1pomocka', 'value')])
def update_page1skenuj(v):
    global my_sniffer_running

    btn_text = 'SKEN - END'
    if my_sniffer_running == 0:
        btn_text = 'SKEN - START'
    elif my_sniffer_running == 1:
        btn_text = 'SKEN - STOP'

    return btn_text


# Callback funkcia pre tlacidlo - "SKEN - START/STOP/END"
@callback(Output('page1pomocka', 'value'),
          Input('page1skenuj', 'n_clicks'),
          prevent_initial_call=True)
def sken_page1pomocka(n_c):
    global my_sniffer
    global my_sniffer_running

    btn_text = '2'
    if my_sniffer_running == 0:
        print("Štart sliedenia...")
        my_sniffer.start()  # spustenie sliedenia v samostatnom vlakne
        my_sniffer_running = 1
        btn_text = '1'
    elif my_sniffer_running == 1:
        print("Stop sliedenia...")
        my_sniffer.join(3)  # ukoncenie vlakna
        my_sniffer_running = 2
        btn_text = '2'

    return btn_text


# Callback funkcia pre tabulku 1 - aktualizacia kazdych 5s
@callback(Output('page1table1', 'data'),
          [Input('page1table1_interval', 'n_intervals')])
def update_page1table1(n):
    # print(pkt_data.to_dict('records'))
    return pkt_data.to_dict('records')


# Callback funkcia pre progress bar - aktualizacia kazdu 1s
@callback(Output('page1progbar', 'value'),
          [Input('page1progbar-interval', 'n_intervals')])
def update_page1progress(n):
    global my_sniffer_running
    global my_sniffer_progress

    if my_sniffer_running == 1:
        my_sniffer_progress += 1
        if my_sniffer_progress > 100:
            my_sniffer_progress = 0

    return my_sniffer_progress
